(function($) {
    'use strict';

    const showNotification = (message, type = 'success') => {
        const notificationArea = $('#notification-area');
        const notice = $('<div>')
            .addClass(`notice notice-${type} is-dismissible`)
            .html(`<p>${message}</p>`);

        notificationArea.append(notice);

        setTimeout(() => {
            notice.fadeOut(() => notice.remove());
        }, 5000);
    };

    // Generate API Key
    $('#generate-key-btn').on('click', function() {
        const button = $(this);
        button.prop('disabled', true).text('Generating...');

        $.ajax({
            url: upperrankData.restUrl + '/generate-key',
            method: 'POST',
            beforeSend: function(xhr) {
                xhr.setRequestHeader('X-WP-Nonce', upperrankData.nonce);
            },
            success: function(response) {
                if (response.success) {
                    $('#api-key-value').text(response.api_key);
                    $('#api-key-result').removeClass('hidden');
                    showNotification('API key generated successfully!', 'success');

                    // Store in hidden button for copy functionality
                    $('#copy-key-btn').data('key', response.api_key);
                }
            },
            error: function(xhr) {
                showNotification('Failed to generate API key. Please try again.', 'error');
                button.prop('disabled', false).text('Generate API Key');
            }
        });
    });

    // Copy API Key
    $(document).on('click', '#copy-key-btn', function() {
        const button = $(this);
        const apiKey = button.data('key') || $('#api-key-value').text();

        if (!apiKey) {
            showNotification('No API key to copy', 'error');
            return;
        }

        // Create temporary textarea
        const textarea = $('<textarea>')
            .val(apiKey)
            .css({
                position: 'fixed',
                top: 0,
                left: 0,
                opacity: 0
            })
            .appendTo('body');

        textarea[0].select();
        document.execCommand('copy');
        textarea.remove();

        const originalText = button.text();
        button.text('Copied!');
        showNotification('API key copied to clipboard!', 'success');

        setTimeout(() => {
            button.text(originalText);
        }, 2000);
    });

    // Regenerate API Key
    $('#regenerate-key-btn').on('click', function() {
        if (!confirm('Are you sure you want to regenerate the API key? The old key will stop working immediately.')) {
            return;
        }

        const button = $(this);
        button.prop('disabled', true).text('Regenerating...');

        $.ajax({
            url: upperrankData.restUrl + '/generate-key',
            method: 'POST',
            beforeSend: function(xhr) {
                xhr.setRequestHeader('X-WP-Nonce', upperrankData.nonce);
            },
            success: function(response) {
                if (response.success) {
                    showNotification('API key regenerated successfully! Please update the key in your UpperRank dashboard.', 'success');

                    setTimeout(() => {
                        location.reload();
                    }, 2000);
                }
            },
            error: function(xhr) {
                showNotification('Failed to regenerate API key. Please try again.', 'error');
                button.prop('disabled', false).text('Regenerate Key');
            }
        });
    });

    // Test Connection
    $('#test-connection-btn').on('click', function() {
        const button = $(this);
        const apiKey = button.closest('.upperrank-card').find('#copy-key-btn').data('key');

        if (!apiKey) {
            showNotification('No API key found. Please generate an API key first.', 'error');
            return;
        }

        button.prop('disabled', true).text('Testing...');

        $.ajax({
            url: upperrankData.restUrl + '/health',
            method: 'GET',
            beforeSend: function(xhr) {
                xhr.setRequestHeader('Authorization', 'Bearer ' + apiKey);
            },
            success: function(response) {
                if (response.success) {
                    showNotification('✅ Connection test successful! Your WordPress site is connected to UpperRank.', 'success');
                }
                button.prop('disabled', false).text('Test Connection');
            },
            error: function(xhr) {
                let message = 'Connection test failed. ';
                if (xhr.responseJSON && xhr.responseJSON.message) {
                    message += xhr.responseJSON.message;
                } else {
                    message += 'Please check your API key and try again.';
                }
                showNotification(message, 'error');
                button.prop('disabled', false).text('Test Connection');
            }
        });
    });

    // Disconnect
    $('#disconnect-btn').on('click', function() {
        if (!confirm('Are you sure you want to disconnect from UpperRank? You will need to reconnect using a new API key.')) {
            return;
        }

        const button = $(this);
        button.prop('disabled', true).text('Disconnecting...');

        $.ajax({
            url: upperrankData.restUrl + '/generate-key',
            method: 'POST',
            beforeSend: function(xhr) {
                xhr.setRequestHeader('X-WP-Nonce', upperrankData.nonce);
            },
            success: function(response) {
                showNotification('Disconnected from UpperRank. Please generate a new API key to reconnect.', 'info');

                setTimeout(() => {
                    location.reload();
                }, 2000);
            },
            error: function(xhr) {
                showNotification('Failed to disconnect. Please try again.', 'error');
                button.prop('disabled', false).text('Disconnect');
            }
        });
    });

    // Save Settings
    $('#upperrank-settings-form').on('submit', function(e) {
        e.preventDefault();

        const button = $(this).find('button[type="submit"]');
        const originalText = button.text();
        button.prop('disabled', true).text('Saving...');

        const formData = {
            default_category: $('#default-category').val(),
            default_author: $('#default-author').val()
        };

        $.ajax({
            url: upperrankData.restUrl + '/settings',
            method: 'POST',
            beforeSend: function(xhr) {
                xhr.setRequestHeader('X-WP-Nonce', upperrankData.nonce);
            },
            data: formData,
            success: function(response) {
                if (response.success) {
                    showNotification('Settings saved successfully!', 'success');
                }
                button.prop('disabled', false).text(originalText);
            },
            error: function(xhr) {
                showNotification('Failed to save settings. Please try again.', 'error');
                button.prop('disabled', false).text(originalText);
            }
        });
    });

})(jQuery);
