<?php
/**
 * Plugin Name: UpperRank Auto Publish
 * Plugin URI: https://upperrank.co/wordpress-plugin
 * Description: Connect your WordPress site to UpperRank for seamless content publishing and scheduling.
 * Version: 2.0.0
 * Author: UpperRank
 * Author URI: https://upperrank.co
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: upperrank-auto-publish
 * Requires at least: 5.0
 * Requires PHP: 7.4
 * Tested up to: 6.8
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Plugin constants
define('UPPERRANK_VERSION', '2.0.0');
define('UPPERRANK_PLUGIN_FILE', __FILE__);
define('UPPERRANK_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('UPPERRANK_PLUGIN_URL', plugin_dir_url(__FILE__));

/**
 * Main UpperRank Plugin Class
 */
class UpperRank_Plugin {
    private static $instance = null;

    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        register_activation_hook(UPPERRANK_PLUGIN_FILE, [$this, 'activate']);
        register_deactivation_hook(UPPERRANK_PLUGIN_FILE, [$this, 'deactivate']);

        add_action('plugins_loaded', [$this, 'init']);
        add_action('rest_api_init', [$this, 'register_rest_routes']);
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_scripts']);
    }

    /**
     * Plugin activation
     */
    public function activate() {
        // Create database table for API keys
        global $wpdb;
        $table_name = $wpdb->prefix . 'upperrank_connections';
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            api_key varchar(64) NOT NULL,
            site_url varchar(255) NOT NULL,
            site_name varchar(255) NOT NULL,
            status varchar(20) DEFAULT 'active',
            last_sync datetime DEFAULT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY api_key (api_key)
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);

        // Set default options
        add_option('upperrank_api_key', '');
        add_option('upperrank_connected', false);
        add_option('upperrank_default_category', '');
        add_option('upperrank_default_author', get_current_user_id());
        add_option('upperrank_version', UPPERRANK_VERSION);
    }

    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Clear scheduled events if any
        wp_clear_scheduled_hook('upperrank_health_check');
    }

    /**
     * Initialize plugin
     */
    public function init() {
        // Translations are automatically loaded by WordPress.org for hosted plugins
    }

    /**
     * Register REST API routes
     */
    public function register_rest_routes() {
        // Generate API key endpoint
        register_rest_route('upperrank/v1', '/generate-key', [
            'methods' => 'POST',
            'callback' => [$this, 'generate_api_key'],
            'permission_callback' => function() {
                return current_user_can('manage_options');
            }
        ]);

        // Connect endpoint
        register_rest_route('upperrank/v1', '/connect', [
            'methods' => 'POST',
            'callback' => [$this, 'connect_site'],
            'permission_callback' => function() {
                return current_user_can('manage_options');
            },
            'args' => [
                'site_url' => [
                    'required' => true,
                    'type' => 'string',
                    'sanitize_callback' => 'esc_url_raw'
                ],
                'site_name' => [
                    'required' => true,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field'
                ]
            ]
        ]);

        // Health check endpoint
        register_rest_route('upperrank/v1', '/health', [
            'methods' => 'GET',
            'callback' => [$this, 'health_check'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Create/update post endpoint
        register_rest_route('upperrank/v1', '/post', [
            'methods' => 'POST',
            'callback' => [$this, 'create_or_update_post'],
            'permission_callback' => [$this, 'verify_api_key'],
            'args' => [
                'title' => [
                    'required' => true,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field'
                ],
                'content' => [
                    'required' => true,
                    'type' => 'string',
                    'sanitize_callback' => 'wp_kses_post'
                ],
                'status' => [
                    'required' => false,
                    'type' => 'string',
                    'default' => 'draft',
                    'sanitize_callback' => 'sanitize_text_field'
                ],
                'scheduled_date' => [
                    'required' => false,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field'
                ],
                'excerpt' => [
                    'required' => false,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_textarea_field'
                ],
                'featured_image_url' => [
                    'required' => false,
                    'type' => 'string',
                    'sanitize_callback' => 'esc_url_raw'
                ],
                'categories' => [
                    'required' => false,
                    'type' => 'array'
                ],
                'tags' => [
                    'required' => false,
                    'type' => 'array'
                ],
                'upperrank_post_id' => [
                    'required' => false,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field'
                ],
                'wordpress_post_id' => [
                    'required' => false,
                    'type' => ['integer', 'null'],
                    'sanitize_callback' => function($value) {
                        return $value ? absint($value) : null;
                    },
                    'validate_callback' => function($value) {
                        return is_null($value) || is_numeric($value);
                    }
                ]
            ]
        ]);

        // Delete post endpoint
        register_rest_route('upperrank/v1', '/post/(?P<id>\d+)', [
            'methods' => 'DELETE',
            'callback' => [$this, 'delete_post'],
            'permission_callback' => [$this, 'verify_api_key']
        ]);

        // Settings endpoint
        register_rest_route('upperrank/v1', '/settings', [
            'methods' => 'POST',
            'callback' => [$this, 'save_settings'],
            'permission_callback' => function() {
                return current_user_can('manage_options');
            },
            'args' => [
                'default_category' => [
                    'required' => false,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field'
                ],
                'default_author' => [
                    'required' => false,
                    'type' => 'integer',
                    'sanitize_callback' => 'absint'
                ]
            ]
        ]);

        // Get settings endpoint
        register_rest_route('upperrank/v1', '/settings', [
            'methods' => 'GET',
            'callback' => [$this, 'get_settings'],
            'permission_callback' => function() {
                return current_user_can('manage_options');
            }
        ]);
    }

    /**
     * Verify API key from request
     */
    public function verify_api_key($request) {
        $auth_header = $request->get_header('Authorization');

        if (empty($auth_header)) {
            return new WP_Error('unauthorized', 'Authorization header is required', ['status' => 401]);
        }

        // Extract API key from Bearer token
        if (strpos($auth_header, 'Bearer ') !== 0) {
            return new WP_Error('unauthorized', 'Invalid authorization format', ['status' => 401]);
        }

        $api_key = substr($auth_header, 7);
        $stored_key = get_option('upperrank_api_key');

        if (empty($stored_key) || !hash_equals($stored_key, $api_key)) {
            return new WP_Error('unauthorized', 'Invalid API key', ['status' => 401]);
        }

        // Update last sync time
        global $wpdb;
        $table_name = $wpdb->prefix . 'upperrank_connections';
        $result = $wpdb->update(
            $table_name,
            ['last_sync' => current_time('mysql')],
            ['api_key' => $api_key],
            ['%s'],
            ['%s']
        );

        // Clear cache after update
        if ($result !== false) {
            wp_cache_delete('upperrank_connection_' . md5($api_key), 'upperrank');
        }

        return true;
    }

    /**
     * Generate new API key
     */
    public function generate_api_key($request) {
        $api_key = wp_generate_password(64, false);
        update_option('upperrank_api_key', $api_key);

        return new WP_REST_Response([
            'success' => true,
            'api_key' => $api_key,
            'message' => 'API key generated successfully'
        ], 200);
    }

    /**
     * Connect site to UpperRank
     */
    public function connect_site($request) {
        $site_url = $request->get_param('site_url');
        $site_name = $request->get_param('site_name');
        $api_key = get_option('upperrank_api_key');

        if (empty($api_key)) {
            // Generate API key if not exists
            $api_key = wp_generate_password(64, false);
            update_option('upperrank_api_key', $api_key);
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'upperrank_connections';

        // Check if connection exists
        $existing = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}upperrank_connections WHERE api_key = %s",
            $api_key
        ));

        if ($existing) {
            // Update existing connection
            $wpdb->update(
                $table_name,
                [
                    'site_url' => $site_url,
                    'site_name' => $site_name,
                    'status' => 'active',
                    'last_sync' => current_time('mysql')
                ],
                ['api_key' => $api_key],
                ['%s', '%s', '%s', '%s'],
                ['%s']
            );
        } else {
            // Insert new connection
            $wpdb->insert(
                $table_name,
                [
                    'api_key' => $api_key,
                    'site_url' => $site_url,
                    'site_name' => $site_name,
                    'status' => 'active',
                    'last_sync' => current_time('mysql')
                ],
                ['%s', '%s', '%s', '%s', '%s']
            );
        }

        update_option('upperrank_connected', true);

        return new WP_REST_Response([
            'success' => true,
            'message' => 'Site connected successfully'
        ], 200);
    }

    /**
     * Health check endpoint
     */
    public function health_check($request) {
        return new WP_REST_Response([
            'success' => true,
            'status' => 'connected',
            'wordpress_version' => get_bloginfo('version'),
            'plugin_version' => UPPERRANK_VERSION,
            'site_url' => home_url(),
            'site_name' => get_bloginfo('name'),
            'timestamp' => current_time('c')
        ], 200);
    }

    /**
     * Create or update post
     */
    public function create_or_update_post($request) {
        $title = $request->get_param('title');
        $content = $request->get_param('content');
        $status = $request->get_param('status') ?: 'draft';
        $excerpt = $request->get_param('excerpt');
        $scheduled_date = $request->get_param('scheduled_date');
        $featured_image_url = $request->get_param('featured_image_url');
        $categories = $request->get_param('categories');
        $tags = $request->get_param('tags');
        $upperrank_post_id = $request->get_param('upperrank_post_id');
        $wordpress_post_id = $request->get_param('wordpress_post_id');

        // Determine if update or create
        $post_id = null;

        if ($wordpress_post_id) {
            $post_id = $wordpress_post_id;
        } elseif ($upperrank_post_id) {
            // Try to find by UpperRank ID
            $posts = get_posts([
                'meta_key' => '_upperrank_post_id',
                'meta_value' => $upperrank_post_id,
                'post_type' => 'post',
                'post_status' => 'any',
                'numberposts' => 1,
                'fields' => 'ids'
            ]);
            if (!empty($posts)) {
                $post_id = $posts[0];
            }
        }

        // Get default author
        $default_author = get_option('upperrank_default_author', get_current_user_id());

        // Convert markdown to HTML if needed
        $html_content = $this->markdown_to_html($content);

        // Prepare post data
        $post_data = [
            'post_title' => $title,
            'post_content' => $html_content,
            'post_status' => $status,
            'post_type' => 'post',
            'post_author' => $default_author
        ];

        if ($excerpt) {
            $post_data['post_excerpt'] = $excerpt;
        }

        if ($scheduled_date && $status === 'future') {
            $post_data['post_date'] = $scheduled_date;
            $post_data['post_date_gmt'] = get_gmt_from_date($scheduled_date);
        }

        if ($post_id) {
            // Update existing post
            $post_data['ID'] = $post_id;
            $result = wp_update_post($post_data, true);
        } else {
            // Create new post
            $result = wp_insert_post($post_data, true);
        }

        if (is_wp_error($result)) {
            return new WP_REST_Response([
                'success' => false,
                'message' => $result->get_error_message()
            ], 400);
        }

        $post_id = $result;

        // Store UpperRank post ID
        if ($upperrank_post_id) {
            update_post_meta($post_id, '_upperrank_post_id', $upperrank_post_id);
        }

        // Handle categories
        $category_ids = [];

        // Add default category if set
        $default_category = get_option('upperrank_default_category');
        if ($default_category) {
            $cat = get_category($default_category);
            if ($cat && !is_wp_error($cat)) {
                $category_ids[] = $cat->term_id;
            }
        }

        // Add categories from request
        if ($categories && is_array($categories)) {
            foreach ($categories as $cat_name) {
                $cat = get_category_by_slug(sanitize_title($cat_name));
                if (!$cat) {
                    $cat_id = wp_insert_category(['cat_name' => $cat_name]);
                    if (!is_wp_error($cat_id)) {
                        $category_ids[] = $cat_id;
                    }
                } else {
                    $category_ids[] = $cat->term_id;
                }
            }
        }

        if (!empty($category_ids)) {
            wp_set_post_categories($post_id, $category_ids);
        }

        // Handle tags
        if ($tags && is_array($tags)) {
            wp_set_post_tags($post_id, $tags);
        }

        // Handle featured image
        if ($featured_image_url) {
            $this->set_featured_image_from_url($post_id, $featured_image_url);
        }

        return new WP_REST_Response([
            'success' => true,
            'wordpress_post_id' => $post_id,
            'post_url' => get_permalink($post_id),
            'edit_url' => esc_url(admin_url('post.php?post=' . $post_id . '&action=edit')),
            'message' => $wordpress_post_id ? 'Post updated successfully' : 'Post created successfully'
        ], 200);
    }

    /**
     * Delete post
     */
    public function delete_post($request) {
        $post_id = $request->get_param('id');

        $post = get_post($post_id);
        if (!$post || $post->post_type !== 'post') {
            return new WP_REST_Response([
                'success' => false,
                'message' => 'Post not found'
            ], 404);
        }

        $result = wp_delete_post($post_id, true);

        if (!$result) {
            return new WP_REST_Response([
                'success' => false,
                'message' => 'Failed to delete post'
            ], 400);
        }

        return new WP_REST_Response([
            'success' => true,
            'message' => 'Post deleted successfully'
        ], 200);
    }

    /**
     * Convert markdown to HTML
     */
    private function markdown_to_html($markdown) {
        if (empty($markdown)) {
            return '';
        }

        $html = $markdown;

        // Escape special HTML characters first
        $html = htmlspecialchars($html, ENT_NOQUOTES, 'UTF-8');

        // Code blocks first (before other processing)
        $html = preg_replace('/```[a-z]*\n(.*?)\n```/s', '<pre><code>$1</code></pre>', $html);

        // Inline code
        $html = preg_replace('/`([^`]+)`/', '<code>$1</code>', $html);

        // Headers (process from most specific to least)
        $html = preg_replace('/^#### (.*?)$/m', '<h4>$1</h4>', $html);
        $html = preg_replace('/^### (.*?)$/m', '<h3>$1</h3>', $html);
        $html = preg_replace('/^## (.*?)$/m', '<h2>$1</h2>', $html);
        $html = preg_replace('/^# (.*?)$/m', '<h1>$1</h1>', $html);

        // Bold (process before italic to avoid conflicts)
        $html = preg_replace('/\*\*(.+?)\*\*/', '<strong>$1</strong>', $html);
        $html = preg_replace('/__(.+?)__/', '<strong>$1</strong>', $html);

        // Italic
        $html = preg_replace('/\*(.+?)\*/', '<em>$1</em>', $html);
        $html = preg_replace('/_(.+?)_/', '<em>$1</em>', $html);

        // Links
        $html = preg_replace('/\[([^\]]+)\]\(([^\)]+)\)/', '<a href="$2" target="_blank" rel="noopener">$1</a>', $html);

        // Images (excluding already processed ones in code blocks)
        $html = preg_replace('/!\[([^\]]*)\]\(([^\)]+)\)/', '<img src="$2" alt="$1" class="wp-image" />', $html);

        // Horizontal rules
        $html = preg_replace('/^---$/m', '<hr />', $html);
        $html = preg_replace('/^\*\*\*$/m', '<hr />', $html);

        // Lists - Unordered
        $html = preg_replace('/^[\*\-\+] (.*)$/m', '<li>$1</li>', $html);

        // Lists - Ordered
        $html = preg_replace('/^\d+\. (.*)$/m', '<li>$1</li>', $html);

        // Wrap list items
        $html = preg_replace('/(<li>.*<\/li>\n)+/s', '<ul>$0</ul>', $html);

        // Blockquotes
        $html = preg_replace('/^> (.*)$/m', '<blockquote>$1</blockquote>', $html);

        // Convert line breaks to paragraphs
        $lines = explode("\n\n", $html);
        $formatted = array();

        foreach ($lines as $line) {
            $line = trim($line);
            if (empty($line)) {
                continue;
            }

            // Check if line is already a block-level element
            if (preg_match('/^<(h[1-6]|ul|ol|blockquote|pre|hr|div|table)/', $line)) {
                $formatted[] = $line;
            } else {
                // Wrap in paragraph tags
                $formatted[] = '<p>' . $line . '</p>';
            }
        }

        $html = implode("\n\n", $formatted);

        // Unescape HTML entities that should be rendered
        $html = str_replace('&lt;', '<', $html);
        $html = str_replace('&gt;', '>', $html);

        return $html;
    }

    /**
     * Set featured image from URL
     */
    private function set_featured_image_from_url($post_id, $image_url) {
        if (empty($image_url)) {
            return false;
        }

        require_once(ABSPATH . 'wp-admin/includes/media.php');
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/image.php');

        $temp_file = download_url($image_url);

        if (is_wp_error($temp_file)) {
            return false;
        }

        // Get file extension from URL
        $parsed_url = wp_parse_url($image_url);
        $file_extension = isset($parsed_url['path']) ? pathinfo($parsed_url['path'], PATHINFO_EXTENSION) : '';
        $file_name = 'upperrank-' . time() . '.' . ($file_extension ?: 'jpg');

        $file_array = [
            'name' => $file_name,
            'tmp_name' => $temp_file
        ];

        $attachment_id = media_handle_sideload($file_array, $post_id);

        if (is_wp_error($attachment_id)) {
            wp_delete_file($temp_file);
            return false;
        }

        set_post_thumbnail($post_id, $attachment_id);
        return true;
    }

    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_menu_page(
            'UpperRank',
            'UpperRank',
            'manage_options',
            'upperrank',
            [$this, 'render_admin_page'],
            'dashicons-admin-links',
            30
        );
    }

    /**
     * Enqueue admin scripts
     */
    public function enqueue_admin_scripts($hook) {
        if ($hook !== 'toplevel_page_upperrank') {
            return;
        }

        wp_enqueue_style('upperrank-admin', UPPERRANK_PLUGIN_URL . 'assets/admin.css', [], UPPERRANK_VERSION);
        wp_enqueue_script('upperrank-admin', UPPERRANK_PLUGIN_URL . 'assets/admin.js', ['jquery'], UPPERRANK_VERSION, true);

        wp_localize_script('upperrank-admin', 'upperrankData', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'restUrl' => rest_url('upperrank/v1'),
            'nonce' => wp_create_nonce('wp_rest'),
            'siteUrl' => home_url(),
            'siteName' => get_bloginfo('name')
        ]);
    }

    /**
     * Save settings
     */
    public function save_settings($request) {
        $default_category = $request->get_param('default_category');
        $default_author = $request->get_param('default_author');

        if ($default_category !== null) {
            update_option('upperrank_default_category', $default_category);
        }

        if ($default_author !== null) {
            update_option('upperrank_default_author', $default_author);
        }

        return new WP_REST_Response([
            'success' => true,
            'message' => 'Settings saved successfully'
        ], 200);
    }

    /**
     * Get settings
     */
    public function get_settings($request) {
        return new WP_REST_Response([
            'default_category' => get_option('upperrank_default_category', ''),
            'default_author' => get_option('upperrank_default_author', get_current_user_id()),
        ], 200);
    }

    /**
     * Render admin page
     */
    public function render_admin_page() {
        $api_key = get_option('upperrank_api_key');
        $connected = get_option('upperrank_connected');
        $default_category = get_option('upperrank_default_category', '');
        $default_author = get_option('upperrank_default_author', get_current_user_id());

        // Get all categories
        $categories = get_categories(['hide_empty' => false]);

        // Get all authors
        $authors = get_users(['who' => 'authors']);

        // Try to get from cache first
        $cache_key = 'upperrank_connection_' . md5($api_key);
        $connection = wp_cache_get($cache_key, 'upperrank');

        if (false === $connection) {
            global $wpdb;
            $table_name = $wpdb->prefix . 'upperrank_connections';
            $connection = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM {$wpdb->prefix}upperrank_connections WHERE api_key = %s ORDER BY created_at DESC LIMIT 1",
                $api_key
            ));

            // Cache the result for 1 hour
            if ($connection) {
                wp_cache_set($cache_key, $connection, 'upperrank', HOUR_IN_SECONDS);
            }
        }

        include UPPERRANK_PLUGIN_DIR . 'admin-page.php';
    }
}

// Initialize the plugin
UpperRank_Plugin::get_instance();
